/* GSequencer - Advanced GTK Sequencer
 * Copyright (C) 2005-2024 Joël Krähemann
 *
 * This file is part of GSequencer.
 *
 * GSequencer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GSequencer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GSequencer.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ags/audio/ags_star_synth_util.h>

#include <ags/audio/ags_audio_buffer_util.h>

#if defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
#include <Accelerate/Accelerate.h>
#endif

#include <math.h>
#include <complex.h>

/**
 * SECTION:ags_star_synth_util
 * @short_description: frequency modulation synth util
 * @title: AgsStarSynthUtil
 * @section_id:
 * @include: ags/audio/ags_star_synth_util.h
 *
 * Utility functions to compute FM synths with vibrato.
 */

GType
ags_star_synth_util_get_type(void)
{
  static gsize g_define_type_id__static = 0;

  if(g_once_init_enter(&g_define_type_id__static)){
    GType ags_type_star_synth_util = 0;

    ags_type_star_synth_util =
      g_boxed_type_register_static("AgsStarSynthUtil",
				   (GBoxedCopyFunc) ags_star_synth_util_copy,
				   (GBoxedFreeFunc) ags_star_synth_util_free);

    g_once_init_leave(&g_define_type_id__static, ags_type_star_synth_util);
  }

  return(g_define_type_id__static);
}

/**
 * ags_star_synth_util_alloc:
 *
 * Allocate #AgsStarSynthUtil-struct
 *
 * Returns: a new #AgsStarSynthUtil-struct
 *
 * Since: 7.4.0
 */
AgsStarSynthUtil*
ags_star_synth_util_alloc()
{
  AgsStarSynthUtil *ptr;

  ptr = (AgsStarSynthUtil *) g_new(AgsStarSynthUtil,
				   1);

  ptr[0] = AGS_STAR_SYNTH_UTIL_INITIALIZER;

  return(ptr);
}

/**
 * ags_star_synth_util_copy:
 * @ptr: the original #AgsStarSynthUtil-struct
 *
 * Create a copy of @ptr.
 *
 * Returns: a pointer of the new #AgsStarSynthUtil-struct
 *
 * Since: 7.4.0
 */
gpointer
ags_star_synth_util_copy(AgsStarSynthUtil *ptr)
{
  AgsStarSynthUtil *new_ptr;

  g_return_val_if_fail(ptr != NULL, NULL);

  new_ptr = (AgsStarSynthUtil *) g_new(AgsStarSynthUtil,
				       1);
  
  new_ptr->source = ptr->source;
  new_ptr->source_stride = ptr->source_stride;

  new_ptr->buffer_length = ptr->buffer_length;
  new_ptr->format = ptr->format;
  new_ptr->samplerate = ptr->samplerate;

  new_ptr->synth_oscillator_mode = ptr->synth_oscillator_mode;

  new_ptr->frequency = ptr->frequency;
  new_ptr->phase = ptr->phase;
  new_ptr->volume = ptr->volume;

  new_ptr->lfo_oscillator_mode = ptr->lfo_oscillator_mode;

  new_ptr->lfo_frequency = ptr->lfo_frequency;
  new_ptr->lfo_depth = ptr->lfo_depth;
  new_ptr->tuning = ptr->tuning;

  new_ptr->vibrato_enabled = ptr->vibrato_enabled;

  new_ptr->vibrato_gain = ptr->vibrato_gain;
  new_ptr->vibrato_lfo_depth = ptr->vibrato_lfo_depth;
  new_ptr->vibrato_lfo_freq = ptr->vibrato_lfo_freq;
  new_ptr->vibrato_tuning = ptr->vibrato_tuning;

  new_ptr->frame_count = ptr->frame_count;
  new_ptr->offset = ptr->offset;

  new_ptr->note_256th_mode = ptr->note_256th_mode;

  new_ptr->offset_256th = ptr->offset_256th;
  
  return(new_ptr);
}

/**
 * ags_star_synth_util_free:
 * @ptr: the #AgsStarSynthUtil-struct
 *
 * Free the memory of @ptr.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_free(AgsStarSynthUtil *ptr)
{
  g_return_if_fail(ptr != NULL);
  
  g_free(ptr);
}

/**
 * ags_star_synth_util_get_source:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get source buffer of @star_synth_util.
 * 
 * Returns: the source buffer
 * 
 * Since: 7.4.0
 */
gpointer
ags_star_synth_util_get_source(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(NULL);
  }

  return(star_synth_util->source);
}

/**
 * ags_star_synth_util_set_source:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @source: (transfer none): the source buffer
 *
 * Set @source buffer of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_source(AgsStarSynthUtil *star_synth_util,
			       gpointer source)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->source = source;
}

/**
 * ags_star_synth_util_get_source_stride:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get source stride of @star_synth_util.
 * 
 * Returns: the source buffer stride
 * 
 * Since: 7.4.0
 */
guint
ags_star_synth_util_get_source_stride(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(0);
  }

  return(star_synth_util->source_stride);
}

/**
 * ags_star_synth_util_set_source_stride:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @source_stride: the source buffer stride
 *
 * Set @source stride of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_source_stride(AgsStarSynthUtil *star_synth_util,
				      guint source_stride)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->source_stride = source_stride;
}

/**
 * ags_star_synth_util_get_buffer_length:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get buffer length of @star_synth_util.
 * 
 * Returns: the buffer length
 * 
 * Since: 7.4.0
 */
guint
ags_star_synth_util_get_buffer_length(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(0);
  }

  return(star_synth_util->buffer_length);
}

/**
 * ags_star_synth_util_set_buffer_length:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @buffer_length: the buffer length
 *
 * Set @buffer_length of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_buffer_length(AgsStarSynthUtil *star_synth_util,
				      guint buffer_length)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->buffer_length = buffer_length;
}

/**
 * ags_star_synth_util_get_format:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get audio buffer util format of @star_synth_util.
 * 
 * Returns: the audio buffer util format
 * 
 * Since: 7.4.0
 */
AgsSoundcardFormat
ags_star_synth_util_get_format(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(0);
  }

  return(star_synth_util->format);
}

/**
 * ags_star_synth_util_set_format:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @format: the audio buffer util format
 *
 * Set @format of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_format(AgsStarSynthUtil *star_synth_util,
			       AgsSoundcardFormat format)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->format = format;
}

/**
 * ags_star_synth_util_get_samplerate:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get samplerate of @star_synth_util.
 * 
 * Returns: the samplerate
 * 
 * Since: 7.4.0
 */
guint
ags_star_synth_util_get_samplerate(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(0);
  }

  return(star_synth_util->samplerate);
}

/**
 * ags_star_synth_util_set_samplerate:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @samplerate: the samplerate
 *
 * Set @samplerate of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_samplerate(AgsStarSynthUtil *star_synth_util,
				   guint samplerate)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->samplerate = samplerate;
}

/**
 * ags_star_synth_util_get_synth_oscillator_mode:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get synth oscillator mode of @star_synth_util.
 * 
 * Returns: the synth oscillator mode
 * 
 * Since: 7.4.0
 */
guint
ags_star_synth_util_get_synth_oscillator_mode(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(0);
  }

  return(star_synth_util->synth_oscillator_mode);
}

/**
 * ags_star_synth_util_set_synth_oscillator_mode:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @synth_oscillator_mode: the synth oscillator mode
 *
 * Set @synth_oscillator_mode of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_synth_oscillator_mode(AgsStarSynthUtil *star_synth_util,
					      guint synth_oscillator_mode)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->synth_oscillator_mode = synth_oscillator_mode;
}

/**
 * ags_star_synth_util_get_frequency:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get frequency of @star_synth_util.
 * 
 * Returns: the frequency
 * 
 * Since: 7.4.0
 */
gdouble
ags_star_synth_util_get_frequency(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(1.0);
  }

  return(star_synth_util->frequency);
}

/**
 * ags_star_synth_util_set_frequency:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @frequency: the frequency
 *
 * Set @frequency of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_frequency(AgsStarSynthUtil *star_synth_util,
				  gdouble frequency)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->frequency = frequency;
}

/**
 * ags_star_synth_util_get_phase:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get phase of @star_synth_util.
 * 
 * Returns: the phase
 * 
 * Since: 7.4.0
 */
gdouble
ags_star_synth_util_get_phase(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(1.0);
  }

  return(star_synth_util->phase);
}

/**
 * ags_star_synth_util_set_phase:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @phase: the phase
 *
 * Set @phase of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_phase(AgsStarSynthUtil *star_synth_util,
			      gdouble phase)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->phase = phase;
}

/**
 * ags_star_synth_util_get_volume:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get volume of @star_synth_util.
 * 
 * Returns: the volume
 * 
 * Since: 7.4.0
 */
gdouble
ags_star_synth_util_get_volume(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(1.0);
  }

  return(star_synth_util->volume);
}

/**
 * ags_star_synth_util_set_volume:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @volume: the volume
 *
 * Set @volume of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_volume(AgsStarSynthUtil *star_synth_util,
			       gdouble volume)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->volume = volume;
}

/**
 * ags_star_synth_util_get_lfo_oscillator_mode:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get LFO oscillator mode of @star_synth_util.
 * 
 * Returns: the LFO oscillator mode
 * 
 * Since: 7.4.0
 */
guint
ags_star_synth_util_get_lfo_oscillator_mode(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(0);
  }

  return(star_synth_util->lfo_oscillator_mode);
}

/**
 * ags_star_synth_util_set_lfo_oscillator_mode:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @lfo_oscillator_mode: the LFO oscillator mode
 *
 * Set @lfo_oscillator_mode of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_lfo_oscillator_mode(AgsStarSynthUtil *star_synth_util,
					    guint lfo_oscillator_mode)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->lfo_oscillator_mode = lfo_oscillator_mode;
}

/**
 * ags_star_synth_util_get_lfo_frequency:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get LFO frequency of @star_synth_util.
 * 
 * Returns: the LFO frequency
 * 
 * Since: 7.4.0
 */
gdouble
ags_star_synth_util_get_lfo_frequency(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(1.0);
  }

  return(star_synth_util->lfo_frequency);
}

/**
 * ags_star_synth_util_set_lfo_frequency:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @lfo_frequency: the LFO frequency
 *
 * Set @lfo_frequency of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_lfo_frequency(AgsStarSynthUtil *star_synth_util,
				      gdouble lfo_frequency)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->lfo_frequency = lfo_frequency;
}

/**
 * ags_star_synth_util_get_lfo_depth:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get LFO depth of @star_synth_util.
 * 
 * Returns: the LFO depth
 * 
 * Since: 7.4.0
 */
gdouble
ags_star_synth_util_get_lfo_depth(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(1.0);
  }

  return(star_synth_util->lfo_depth);
}

/**
 * ags_star_synth_util_set_lfo_depth:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @lfo_depth: the LFO depth
 *
 * Set @lfo_depth of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_lfo_depth(AgsStarSynthUtil *star_synth_util,
				  gdouble lfo_depth)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->lfo_depth = lfo_depth;
}

/**
 * ags_star_synth_util_get_tuning:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get tuning of @star_synth_util.
 * 
 * Returns: the tuning
 * 
 * Since: 7.4.0
 */
gdouble
ags_star_synth_util_get_tuning(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(1.0);
  }

  return(star_synth_util->tuning);
}

/**
 * ags_star_synth_util_set_tuning:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @tuning: the tuning
 *
 * Set @tuning of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_tuning(AgsStarSynthUtil *star_synth_util,
			       gdouble tuning)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->tuning = tuning;
}

/**
 * ags_star_synth_util_get_vibrato_enabled:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get vibrato enabled of @star_synth_util.
 * 
 * Returns: %TRUE if the vibrato is enabled, otherwise %FALSE
 * 
 * Since: 7.4.0
 */
gboolean
ags_star_synth_util_get_vibrato_enabled(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(FALSE);
  }

  return(star_synth_util->vibrato_enabled);
}

/**
 * ags_star_synth_util_set_vibrato_enabled:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @vibrato_enabled: the vibrato enabled
 *
 * Set @vibrato_enabled of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_vibrato_enabled(AgsStarSynthUtil *star_synth_util,
					gboolean vibrato_enabled)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->vibrato_enabled = vibrato_enabled;
}

/**
 * ags_star_synth_util_get_vibrato_gain:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get vibrato gain of @star_synth_util.
 * 
 * Returns: the vibrato gain
 * 
 * Since: 7.4.0
 */
gdouble
ags_star_synth_util_get_vibrato_gain(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(0.0);
  }

  return(star_synth_util->vibrato_gain);
}

/**
 * ags_star_synth_util_set_vibrato_gain:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @vibrato_gain: the vibrato gain
 *
 * Set @vibrato_gain of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_vibrato_gain(AgsStarSynthUtil *star_synth_util,
				     gdouble vibrato_gain)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->vibrato_gain = vibrato_gain;
}

/**
 * ags_star_synth_util_get_vibrato_lfo_depth:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get vibrato lfo_depth of @star_synth_util.
 * 
 * Returns: the vibrato lfo_depth
 * 
 * Since: 7.4.0
 */
gdouble
ags_star_synth_util_get_vibrato_lfo_depth(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(0.0);
  }

  return(star_synth_util->vibrato_lfo_depth);
}

/**
 * ags_star_synth_util_set_vibrato_lfo_depth:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @vibrato_lfo_depth: the vibrato lfo_depth
 *
 * Set @vibrato_lfo_depth of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_vibrato_lfo_depth(AgsStarSynthUtil *star_synth_util,
					  gdouble vibrato_lfo_depth)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->vibrato_lfo_depth = vibrato_lfo_depth;
}

/**
 * ags_star_synth_util_get_vibrato_lfo_freq:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get vibrato lfo_freq of @star_synth_util.
 * 
 * Returns: the vibrato lfo_freq
 * 
 * Since: 7.4.0
 */
gdouble
ags_star_synth_util_get_vibrato_lfo_freq(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(0.0);
  }

  return(star_synth_util->vibrato_lfo_freq);
}

/**
 * ags_star_synth_util_set_vibrato_lfo_freq:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @vibrato_lfo_freq: the vibrato lfo_freq
 *
 * Set @vibrato_lfo_freq of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_vibrato_lfo_freq(AgsStarSynthUtil *star_synth_util,
					 gdouble vibrato_lfo_freq)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->vibrato_lfo_freq = vibrato_lfo_freq;
}

/**
 * ags_star_synth_util_get_vibrato_tuning:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get vibrato tuning of @star_synth_util.
 * 
 * Returns: the vibrato tuning
 * 
 * Since: 7.4.0
 */
gdouble
ags_star_synth_util_get_vibrato_tuning(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(0.0);
  }

  return(star_synth_util->vibrato_tuning);
}

/**
 * ags_star_synth_util_set_vibrato_tuning:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @vibrato_tuning: the vibrato tuning
 *
 * Set @vibrato_tuning of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_vibrato_tuning(AgsStarSynthUtil *star_synth_util,
				       gdouble vibrato_tuning)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->vibrato_tuning = vibrato_tuning;
}

/**
 * ags_star_synth_util_get_frame_count:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get frame count of @star_synth_util.
 * 
 * Returns: the frame count
 * 
 * Since: 7.4.0
 */
guint
ags_star_synth_util_get_frame_count(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(0);
  }

  return(star_synth_util->frame_count);
}

/**
 * ags_star_synth_util_set_frame_count:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @frame_count: the frame count
 *
 * Set @frame_count of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_frame_count(AgsStarSynthUtil *star_synth_util,
				    guint frame_count)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->frame_count = frame_count;
}

/**
 * ags_star_synth_util_get_offset:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get offset of @star_synth_util.
 * 
 * Returns: the offset
 * 
 * Since: 7.4.0
 */
guint
ags_star_synth_util_get_offset(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(0);
  }

  return(star_synth_util->offset);
}

/**
 * ags_star_synth_util_set_offset:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @offset: the offset
 *
 * Set @offset of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_offset(AgsStarSynthUtil *star_synth_util,
			       guint offset)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->offset = offset;
}

/**
 * ags_star_synth_util_get_note_256th_mode:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get note 256th mode of @star_synth_util.
 * 
 * Returns: %TRUE if note 256th mode, otherwise %FALSE
 * 
 * Since: 7.4.0
 */
gboolean
ags_star_synth_util_get_note_256th_mode(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(0);
  }

  return(star_synth_util->note_256th_mode);
}

/**
 * ags_star_synth_util_set_note_256th_mode:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @note_256th_mode: the note 256th mode
 *
 * Set @note_256th_mode of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_note_256th_mode(AgsStarSynthUtil *star_synth_util,
					gboolean note_256th_mode)
{
  if(star_synth_util == NULL){
    return;
  }

  star_synth_util->note_256th_mode = note_256th_mode;
}

/**
 * ags_star_synth_util_get_offset_256th:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Get offset as note 256th of @star_synth_util.
 * 
 * Returns: the offset as note 256th
 * 
 * Since: 7.4.0
 */
guint
ags_star_synth_util_get_offset_256th(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL){
    return(0);
  }

  return(star_synth_util->offset_256th);
}

/**
 * ags_star_synth_util_set_offset_256th:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * @offset_256th: the offset as note 256th
 *
 * Set @offset_256th of @star_synth_util.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_set_offset_256th(AgsStarSynthUtil *star_synth_util,
				     guint offset_256th)
{
  if(star_synth_util == NULL){
    return;
  }
  
  star_synth_util->offset_256th = offset_256th;
}

/**
 * ags_star_synth_util_compute_sin_s8:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sine synth of signed 8 bit data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sin_s8(AgsStarSynthUtil *star_synth_util)
{
  gint8 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 127.0;
  
  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sine;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 1 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 2 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 3 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 4 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 5 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 6 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 7 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate)	
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    default:
      v_sine = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sine *= scale;
    v_sine *= volume;

    v_buffer += v_sine;

    *(source) = (gint8) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[7];

    source += star_synth_util->source_stride;    
      
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_sine[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    default:
      v_sine[0] = 0.0;
      v_sine[1] = 0.0;
      v_sine[2] = 0.0;
      v_sine[3] = 0.0;
      v_sine[4] = 0.0;
      v_sine[5] = 0.0;
      v_sine[6] = 0.0;
      v_sine[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sine, 1, v_scale, 0, tmp0_ret_v_buffer, 1, 8);

    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_volume, 0, tmp1_ret_v_buffer, 1, 8);

    vDSP_vaddD(v_buffer, 1, tmp1_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint8) ret_v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint8) ret_v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint8) ret_v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint8) ret_v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint8) ret_v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint8) ret_v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint8) ret_v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint8) ret_v_buffer[7];

    source += star_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }
    
    source += star_synth_util->source_stride;

    i += 8;
  }
#endif

  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }
    
    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sin_s16:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sine synth of signed 16 bit data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sin_s16(AgsStarSynthUtil *star_synth_util)
{
  gint16 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 32767.0;
  
  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sine;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 1 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 2 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 3 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 4 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 5 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 6 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 7 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate)	
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    default:
      v_sine = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sine *= scale;
    v_sine *= volume;

    v_buffer += v_sine;

    *(source) = (gint16) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint16) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint16) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint16) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint16) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint16) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint16) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint16) v_buffer[7];

    source += star_synth_util->source_stride;

    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_sine[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    default:
      v_sine[0] = 0.0;
      v_sine[1] = 0.0;
      v_sine[2] = 0.0;
      v_sine[3] = 0.0;
      v_sine[4] = 0.0;
      v_sine[5] = 0.0;
      v_sine[6] = 0.0;
      v_sine[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sine, 1, v_scale, 0, tmp0_ret_v_buffer, 1, 8);
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_volume, 0, tmp1_ret_v_buffer, 1, 8);
    
    vDSP_vaddD(v_buffer, 1, tmp1_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint16) ret_v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint16) ret_v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint16) ret_v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint16) ret_v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint16) ret_v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint16) ret_v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint16) ret_v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint16) ret_v_buffer[7];

    source += star_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }

    source += star_synth_util->source_stride;
    i += 8;
  }
#endif

  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }

    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sin_s24:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sine synth of signed 24 bit data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sin_s24(AgsStarSynthUtil *star_synth_util)
{
  gint32 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 8388607.0;
  
  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sine;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate)
	};      
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 1 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 2 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 3 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 4 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 5 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 6 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 7 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate)	
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    default:
      v_sine = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sine *= scale;
    v_sine *= volume;

    v_buffer += v_sine;

    *(source) = (gint32) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[7];

    source += star_synth_util->source_stride;
    
    i += 8; 
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_sine[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    default:
      v_sine[0] = 0.0;
      v_sine[1] = 0.0;
      v_sine[2] = 0.0;
      v_sine[3] = 0.0;
      v_sine[4] = 0.0;
      v_sine[5] = 0.0;
      v_sine[6] = 0.0;
      v_sine[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sine, 1, v_scale, 0, tmp0_ret_v_buffer, 1, 8);
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_volume, 0, tmp1_ret_v_buffer, 1, 8);
    vDSP_vaddD(v_buffer, 1, tmp1_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint32) ret_v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[7];

    source += star_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }

    source += star_synth_util->source_stride;
    i += 8;
  }
#endif

  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }
    
    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sin_s32:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sine synth of signed 32 bit data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sin_s32(AgsStarSynthUtil *star_synth_util)
{
  gint32 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 214748363.0;
  
  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sine;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 1 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 2 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 3 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 4 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 5 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 6 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 7 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate)	
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)
	};    
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};    
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    default:
      v_sine = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sine *= scale;
    v_sine *= volume;

    v_buffer += v_sine;

    *(source) = (gint32) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[7];

    source += star_synth_util->source_stride;    
      
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_sine[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    default:
      v_sine[0] = 0.0;
      v_sine[1] = 0.0;
      v_sine[2] = 0.0;
      v_sine[3] = 0.0;
      v_sine[4] = 0.0;
      v_sine[5] = 0.0;
      v_sine[6] = 0.0;
      v_sine[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sine, 1, v_scale, 0, tmp0_ret_v_buffer, 1, 8);
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_volume, 0, tmp1_ret_v_buffer, 1, 8);
    vDSP_vaddD(v_buffer, 1, tmp1_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint32) ret_v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[7];

    source += star_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }

    source += star_synth_util->source_stride;
    i += 8;
  }
#endif

  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }
    
    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sin_s64:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sine synth of signed 64 bit data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sin_s64(AgsStarSynthUtil *star_synth_util)
{
  gint64 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 9223372036854775807.0;
  
  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sine;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate)
	};      
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 1 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 2 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 3 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 4 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 5 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 6 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 7 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate)	
	};    
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};    
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    default:
      v_sine = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sine *= scale;
    v_sine *= volume;

    v_buffer += v_sine;

    *(source) = (gint64) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint64) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint64) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint64) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint64) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint64) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint64) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint64) v_buffer[7];

    source += star_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_sine[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    default:
      v_sine[0] = 0.0;
      v_sine[1] = 0.0;
      v_sine[2] = 0.0;
      v_sine[3] = 0.0;
      v_sine[4] = 0.0;
      v_sine[5] = 0.0;
      v_sine[6] = 0.0;
      v_sine[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sine, 1, v_scale, 0, tmp0_ret_v_buffer, 1, 8);
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_volume, 0, tmp1_ret_v_buffer, 1, 8);
    vDSP_vaddD(v_buffer, 1, tmp1_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint64) ret_v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint64) ret_v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint64) ret_v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint64) ret_v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint64) ret_v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint64) ret_v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint64) ret_v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint64) ret_v_buffer[7];

    source += star_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) (sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }

    source += star_synth_util->source_stride;
    i += 8;
  }
#endif

  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * scale * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * scale * volume);
      }
      break;
    }

    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sin_float:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sine synth of floating point data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sin_float(AgsStarSynthUtil *star_synth_util)
{
  gfloat *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sine;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate)
	};      
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 1 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 2 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 3 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 4 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 5 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 6 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 7 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate)	
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    default:
      v_sine = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sine *= volume;

    v_buffer += v_sine;

    *(source) = (gfloat) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gfloat) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gfloat) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gfloat) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gfloat) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gfloat) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gfloat) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gfloat) v_buffer[7];

    source += star_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_sine[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    default:
      v_sine[0] = 0.0;
      v_sine[1] = 0.0;
      v_sine[2] = 0.0;
      v_sine[3] = 0.0;
      v_sine[4] = 0.0;
      v_sine[5] = 0.0;
      v_sine[6] = 0.0;
      v_sine[7] = 0.0;
    };

    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sine, 1, v_volume, 0, tmp_ret_v_buffer, 1, 8);
    vDSP_vaddD(v_buffer, 1, tmp_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gfloat) ret_v_buffer[0];
    *(source += star_synth_util->source_stride) = (gfloat) ret_v_buffer[1];
    *(source += star_synth_util->source_stride) = (gfloat) ret_v_buffer[2];
    *(source += star_synth_util->source_stride) = (gfloat) ret_v_buffer[3];
    *(source += star_synth_util->source_stride) = (gfloat) ret_v_buffer[4];
    *(source += star_synth_util->source_stride) = (gfloat) ret_v_buffer[5];
    *(source += star_synth_util->source_stride) = (gfloat) ret_v_buffer[6];
    *(source += star_synth_util->source_stride) = (gfloat) ret_v_buffer[7];

    source += star_synth_util->source_stride;
    
    i += 8;
  }
#else
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	*(source) = (gfloat) ((tmp_source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gfloat) ((tmp_source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gfloat) ((tmp_source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gfloat) ((tmp_source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gfloat) ((tmp_source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    }

    source += star_synth_util->source_stride;

    i += 8;
  }
#endif

  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	*(source) = (gfloat) ((source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gfloat) ((source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gfloat) ((source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gfloat) ((source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gfloat) ((source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    }

    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sin_double:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sine synth of double precision floating point data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sin_double(AgsStarSynthUtil *star_synth_util)
{
  gdouble *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sine;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate)
	};      
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 1 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 2 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 3 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 4 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 5 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 6 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) (offset + i + 7 + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate)	
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine = (ags_v8double) {
	  sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate),
	  sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)
	};
      }
      break;
    default:
      v_sine = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sine *= volume;

    v_buffer += v_sine;

    *(source) = (gdouble) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gdouble) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gdouble) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gdouble) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gdouble) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gdouble) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gdouble) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gdouble) v_buffer[7];

    source += star_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_sine[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sine[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[1] = sin((gdouble) ((offset + (i + 1)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[2] = sin((gdouble) ((offset + (i + 2)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[3] = sin((gdouble) ((offset + (i + 3)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[4] = sin((gdouble) ((offset + (i + 4)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[5] = sin((gdouble) ((offset + (i + 5)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[6] = sin((gdouble) ((offset + (i + 6)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_sine[7] = sin((gdouble) ((offset + (i + 7)) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    default:
      v_sine[0] = 0.0;
      v_sine[1] = 0.0;
      v_sine[2] = 0.0;
      v_sine[3] = 0.0;
      v_sine[4] = 0.0;
      v_sine[5] = 0.0;
      v_sine[6] = 0.0;
      v_sine[7] = 0.0;
    };

    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sine, 1, v_volume, 0, tmp_ret_v_buffer, 1, 8);
    vDSP_vaddD(v_buffer, 1, tmp_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gdouble) ret_v_buffer[0];
    *(source += star_synth_util->source_stride) = (gdouble) ret_v_buffer[1];
    *(source += star_synth_util->source_stride) = (gdouble) ret_v_buffer[2];
    *(source += star_synth_util->source_stride) = (gdouble) ret_v_buffer[3];
    *(source += star_synth_util->source_stride) = (gdouble) ret_v_buffer[4];
    *(source += star_synth_util->source_stride) = (gdouble) ret_v_buffer[5];
    *(source += star_synth_util->source_stride) = (gdouble) ret_v_buffer[6];
    *(source += star_synth_util->source_stride) = (gdouble) ret_v_buffer[7];

    source += star_synth_util->source_stride;
    
    i += 8;
  }
#else
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	*(source) = (gdouble) ((tmp_source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gdouble) ((tmp_source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gdouble) ((tmp_source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gdouble) ((tmp_source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gdouble) ((tmp_source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    }

    source += star_synth_util->source_stride;

    i += 8;
  }
#endif

  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	*(source) = (gdouble) ((source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gdouble) ((source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gdouble) ((source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gdouble) ((source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gdouble) ((source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    }

    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sin_complex:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sine synth of complex data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sin_complex(AgsStarSynthUtil *star_synth_util)
{
  AgsComplex *source;
  
  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i;

  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	ags_complex_set(source,
			(ags_complex_get(source) + (sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / (gdouble) samplerate) * volume)));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	ags_complex_set(source,
			(ags_complex_get(source) + (sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth) / (gdouble) samplerate)) * volume)));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	ags_complex_set(source,
			(ags_complex_get(source) + (sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate) * volume)));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	ags_complex_set(source,
			(ags_complex_get(source) + (sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate) * volume)));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	ags_complex_set(source,
			(ags_complex_get(source) + (sin((gdouble) (offset + i + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate) * volume)));
      }
      break;
    }

    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sin:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sine synth.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sin(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  switch(star_synth_util->format){
  case AGS_SOUNDCARD_SIGNED_8_BIT:
    {
      ags_star_synth_util_compute_sin_s8(star_synth_util);
    }
    break;
  case AGS_SOUNDCARD_SIGNED_16_BIT:
    {
      ags_star_synth_util_compute_sin_s16(star_synth_util);
    }
    break;
  case AGS_SOUNDCARD_SIGNED_24_BIT:
    {
      ags_star_synth_util_compute_sin_s24(star_synth_util);
    }
    break;
  case AGS_SOUNDCARD_SIGNED_32_BIT:
    {
      ags_star_synth_util_compute_sin_s32(star_synth_util);
    }
    break;
  case AGS_SOUNDCARD_SIGNED_64_BIT:
    {
      ags_star_synth_util_compute_sin_s64(star_synth_util);
    }
    break;
  case AGS_SOUNDCARD_FLOAT:
    {
      ags_star_synth_util_compute_sin_float(star_synth_util);
    }
    break;
  case AGS_SOUNDCARD_DOUBLE:
    {
      ags_star_synth_util_compute_sin_double(star_synth_util);
    }
    break;
  case AGS_SOUNDCARD_COMPLEX:
    {
      ags_star_synth_util_compute_sin_complex(star_synth_util);
    }
    break;
  }
}

/**
 * ags_star_synth_util_compute_sawtooth_s8:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sawtooth synth of signed 8 bit data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sawtooth_s8(AgsStarSynthUtil *star_synth_util)
{
  gint8 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 127.0;
  
  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sawtooth;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0)
	};      
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};    
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    default:
      v_sawtooth = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sawtooth *= scale;
    v_sawtooth *= volume;

    v_buffer += v_sawtooth;

    *(source) = (gint8) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[7];

    source += star_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_sawtooth[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    default:
      v_sawtooth[0] = 0.0;
      v_sawtooth[1] = 0.0;
      v_sawtooth[2] = 0.0;
      v_sawtooth[3] = 0.0;
      v_sawtooth[4] = 0.0;
      v_sawtooth[5] = 0.0;
      v_sawtooth[6] = 0.0;
      v_sawtooth[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sawtooth, 1, v_scale, 0, tmp0_ret_v_buffer, 1, 8);
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_volume, 0, tmp1_ret_v_buffer, 1, 8);
    vDSP_vaddD(v_buffer, 1, tmp1_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint8) ret_v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint8) ret_v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint8) ret_v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint8) ret_v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint8) ret_v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint8) ret_v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint8) ret_v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint8) ret_v_buffer[7];

    source += star_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) scale * ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) scale * (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint8) ((gint16) (tmp_source)[0] + (gint16) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint8) ((gint16) (tmp_source += star_synth_util->source_stride)[0] + (gint16) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }
    
    source += star_synth_util->source_stride;

    i += 8;
  }
#endif

  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) scale * ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) scale * (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint8) ((gint16) (source)[0] + (gint16) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }

    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sawtooth_s16:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sawtooth synth of signed 16 bit data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sawtooth_s16(AgsStarSynthUtil *star_synth_util)
{
  gint16 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 32767.0;
  
  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sawtooth;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    default:
      v_sawtooth = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sawtooth *= scale;
    v_sawtooth *= volume;

    v_buffer += v_sawtooth;

    *(source) = (gint16) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint16) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint16) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint16) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint16) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint16) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint16) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint16) v_buffer[7];

    source += star_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_sawtooth[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    default:
      v_sawtooth[0] = 0.0;
      v_sawtooth[1] = 0.0;
      v_sawtooth[2] = 0.0;
      v_sawtooth[3] = 0.0;
      v_sawtooth[4] = 0.0;
      v_sawtooth[5] = 0.0;
      v_sawtooth[6] = 0.0;
      v_sawtooth[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sawtooth, 1, v_scale, 0, tmp0_ret_v_buffer, 1, 8);
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_volume, 0, tmp1_ret_v_buffer, 1, 8);
    vDSP_vaddD(v_buffer, 1, tmp1_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint16) ret_v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint16) ret_v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint16) ret_v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint16) ret_v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint16) ret_v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint16) ret_v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint16) ret_v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint16) ret_v_buffer[7];

    source += star_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) scale * ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) scale * (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint16) ((gint32) (tmp_source)[0] + (gint32) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint16) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }
    
    source += star_synth_util->source_stride;

    i += 8;
  }
#endif

  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) scale * ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) scale * (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint16) ((gint32) (source)[0] + (gint32) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }    

    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sawtooth_s24:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sawtooth synth of signed 24 bit data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sawtooth_s24(AgsStarSynthUtil *star_synth_util)
{
  gint32 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 8388607.0;
  
  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sawtooth;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    default:
      v_sawtooth = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sawtooth *= scale;
    v_sawtooth *= volume;

    v_buffer += v_sawtooth;

    *(source) = (gint32) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[7];

    source += star_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_sawtooth[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    default:
      v_sawtooth[0] = 0.0;
      v_sawtooth[1] = 0.0;
      v_sawtooth[2] = 0.0;
      v_sawtooth[3] = 0.0;
      v_sawtooth[4] = 0.0;
      v_sawtooth[5] = 0.0;
      v_sawtooth[6] = 0.0;
      v_sawtooth[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sawtooth, 1, v_scale, 0, tmp0_ret_v_buffer, 1, 8);
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_volume, 0, tmp1_ret_v_buffer, 1, 8);
    vDSP_vaddD(v_buffer, 1, tmp1_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint32) ret_v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[7];

    source += star_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) scale * ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) scale * (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint32) (tmp_source)[0] + (gint32) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint32) (tmp_source += star_synth_util->source_stride)[0] + (gint32) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }
    
    source += star_synth_util->source_stride;
    i += 8;
  }
#endif

  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) scale * ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) scale * (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint32) (source)[0] + (gint32) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }

    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sawtooth_s32:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sawtooth synth of signed 32 bit data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sawtooth_s32(AgsStarSynthUtil *star_synth_util)
{
  gint32 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 214748363.0;

  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sawtooth;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    default:
      v_sawtooth = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }
    
    v_sawtooth *= scale;
    v_sawtooth *= volume;

    v_buffer += v_sawtooth;
    
    *(source) = (gint32) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint32) v_buffer[7];
    
    source += star_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_sawtooth[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    default:
      v_sawtooth[0] = 0.0;
      v_sawtooth[1] = 0.0;
      v_sawtooth[2] = 0.0;
      v_sawtooth[3] = 0.0;
      v_sawtooth[4] = 0.0;
      v_sawtooth[5] = 0.0;
      v_sawtooth[6] = 0.0;
      v_sawtooth[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};

    vDSP_vmulD(v_sawtooth, 1, v_scale, 0, tmp0_ret_v_buffer, 1, 8);
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_volume, 0, tmp1_ret_v_buffer, 1, 8);
    vDSP_vaddD(v_buffer, 1, tmp1_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint32) ret_v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint32) ret_v_buffer[7];

    source += star_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) scale * ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) scale * (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint64) (tmp_source)[0] + (gint64) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 1)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 2)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 3)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 4)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 5)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 6)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint32) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + (i + 7)) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }
    
    source += star_synth_util->source_stride;

    i += 8;
  }
#endif

  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) scale * ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) scale * (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint32) ((gint64) (source)[0] + (gint64) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }
  
    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sawtooth_s64:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sawtooth synth of signed 64 bit data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sawtooth_s64(AgsStarSynthUtil *star_synth_util)
{
  gint64 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 9223372036854775807.0;
  
  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sawtooth;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0)
	};      
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    default:
      v_sawtooth = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sawtooth *= scale;
    v_sawtooth *= volume;

    v_buffer += v_sawtooth;

    *(source) = (gint64) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint64) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint64) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint64) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint64) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint64) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint64) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint64) v_buffer[7];

    source += star_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_sawtooth[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    default:
      v_sawtooth[0] = 0.0;
      v_sawtooth[1] = 0.0;
      v_sawtooth[2] = 0.0;
      v_sawtooth[3] = 0.0;
      v_sawtooth[4] = 0.0;
      v_sawtooth[5] = 0.0;
      v_sawtooth[6] = 0.0;
      v_sawtooth[7] = 0.0;
    };

    double v_scale[] = {(double) scale};
    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sawtooth, 1, v_scale, 0, tmp0_ret_v_buffer, 1, 8);
    vDSP_vmulD(tmp0_ret_v_buffer, 1, v_volume, 0, tmp1_ret_v_buffer, 1, 8);
    vDSP_vaddD(v_buffer, 1, tmp1_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gint64) ret_v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint64) ret_v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint64) ret_v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint64) ret_v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint64) ret_v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint64) ret_v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint64) ret_v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint64) ret_v_buffer[7];

    source += star_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) scale * ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) scale * (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint64) ((gint64) (tmp_source)[0] + (gint64) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gint64) ((gint64) (tmp_source += star_synth_util->source_stride)[0] + (gint64) scale * (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }
    
    source += star_synth_util->source_stride;
    
    i += 8;
  }
#endif

  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) scale * ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) scale * (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gint64) ((gint64) (source)[0] + (gint64) scale * (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }

    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sawtooth_float:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sawtooth synth of floating point data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sawtooth_float(AgsStarSynthUtil *star_synth_util)
{
  gfloat *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;
  
  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sawtooth;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    default:
      v_sawtooth = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sawtooth *= volume;

    v_buffer += v_sawtooth;

    *(source) = (gfloat) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gfloat) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gfloat) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gfloat) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gfloat) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gfloat) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gfloat) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gfloat) v_buffer[7];

    source += star_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_sawtooth[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    default:
      v_sawtooth[0] = 0.0;
      v_sawtooth[1] = 0.0;
      v_sawtooth[2] = 0.0;
      v_sawtooth[3] = 0.0;
      v_sawtooth[4] = 0.0;
      v_sawtooth[5] = 0.0;
      v_sawtooth[6] = 0.0;
      v_sawtooth[7] = 0.0;
    };

    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sawtooth, 1, v_volume, 0, tmp_ret_v_buffer, 1, 8);
    vDSP_vaddD(v_buffer, 1, tmp_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gfloat) ret_v_buffer[0];
    *(source += star_synth_util->source_stride) = (gfloat) ret_v_buffer[1];
    *(source += star_synth_util->source_stride) = (gfloat) ret_v_buffer[2];
    *(source += star_synth_util->source_stride) = (gfloat) ret_v_buffer[3];
    *(source += star_synth_util->source_stride) = (gfloat) ret_v_buffer[4];
    *(source += star_synth_util->source_stride) = (gfloat) ret_v_buffer[5];
    *(source += star_synth_util->source_stride) = (gfloat) ret_v_buffer[6];
    *(source += star_synth_util->source_stride) = (gfloat) ret_v_buffer[7];

    source += star_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gfloat) ((tmp_source)[0] + (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gfloat) ((tmp_source)[0] + ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gfloat) ((tmp_source)[0] + (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gfloat) ((tmp_source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gfloat) ((tmp_source)[0] + (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gfloat) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }
    
    source += star_synth_util->source_stride;

    i += 8;
  }
#endif

  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gfloat) ((source)[0] + (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gfloat) ((source)[0] + ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gfloat) ((source)[0] + (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gfloat) ((source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gfloat) ((source)[0] + (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }

    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sawtooth_double:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sawtooth synth of double precision floating point data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sawtooth_double(AgsStarSynthUtil *star_synth_util)
{
  gdouble *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;
  
  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_sawtooth;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth = (ags_v8double) {
	  ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0),
	  ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0)
	};
      }
      break;
    default:
      v_sawtooth = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_sawtooth *= volume;

    v_buffer += v_sawtooth;

    *(source) = (gdouble) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gdouble) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gdouble) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gdouble) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gdouble) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gdouble) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gdouble) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gdouble) v_buffer[7];

    source += star_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_sawtooth[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_sawtooth[0] = ((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[1] = ((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[2] = ((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[3] = ((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[4] = ((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[5] = ((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[6] = ((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
	v_sawtooth[7] = ((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0);
      }
      break;
    default:
      v_sawtooth[0] = 0.0;
      v_sawtooth[1] = 0.0;
      v_sawtooth[2] = 0.0;
      v_sawtooth[3] = 0.0;
      v_sawtooth[4] = 0.0;
      v_sawtooth[5] = 0.0;
      v_sawtooth[6] = 0.0;
      v_sawtooth[7] = 0.0;
    };

    double v_volume[] = {(double) volume};
    
    vDSP_vmulD(v_sawtooth, 1, v_volume, 0, tmp_ret_v_buffer, 1, 8);
    vDSP_vaddD(v_buffer, 1, tmp_ret_v_buffer, 1, ret_v_buffer, 1, 8);
    
    *(source) = (gdouble) ret_v_buffer[0];
    *(source += star_synth_util->source_stride) = (gdouble) ret_v_buffer[1];
    *(source += star_synth_util->source_stride) = (gdouble) ret_v_buffer[2];
    *(source += star_synth_util->source_stride) = (gdouble) ret_v_buffer[3];
    *(source += star_synth_util->source_stride) = (gdouble) ret_v_buffer[4];
    *(source += star_synth_util->source_stride) = (gdouble) ret_v_buffer[5];
    *(source += star_synth_util->source_stride) = (gdouble) ret_v_buffer[6];
    *(source += star_synth_util->source_stride) = (gdouble) ret_v_buffer[7];

    source += star_synth_util->source_stride;

    i += 8;
  }
#else
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    tmp_source = source;

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gdouble) ((tmp_source)[0] + (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gdouble) ((tmp_source)[0] + ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gdouble) ((tmp_source)[0] + (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gdouble) ((tmp_source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gdouble) ((tmp_source)[0] + (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 1) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 2) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 3) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 4) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 5) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 6) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
	*(source += star_synth_util->source_stride) = (gdouble) ((tmp_source += star_synth_util->source_stride)[0] + (((((int) ceil((offset + i + 7) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }
    
    source += star_synth_util->source_stride;

    i += 8;
  }
#endif

  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	(*source) = (gdouble) ((source)[0] + (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	(*source) = (gdouble) ((source)[0] + ((sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / (gdouble) samplerate) - 1.0) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	(*source) = (gdouble) ((source)[0] + (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	(*source) = (gdouble) ((source)[0] + (sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate)) * volume);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	(*source) = (gdouble) ((source)[0] + (((((int) ceil((offset + i) + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume));
      }
      break;
    }

    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sawtooth_complex:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sawtooth synth of complex data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sawtooth_complex(AgsStarSynthUtil *star_synth_util)
{
  AgsComplex *source;
  
  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i;

  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
  for(; i < star_synth_util->buffer_length;){
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	ags_complex_set(source,
			(ags_complex_get(source) + (((((int) ceil(offset + i + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate) - 1.0) * volume)));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	ags_complex_set(source,
			(ags_complex_get(source) + (((((int) ceil(offset + i + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate) - 1.0) * volume)));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	ags_complex_set(source,
			(ags_complex_get(source) + (((((int) ceil(offset + i + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate) - 1.0) * volume)));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	ags_complex_set(source,
			(ags_complex_get(source) + (((((int) ceil(offset + i + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume)));
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	ags_complex_set(source,
			(ags_complex_get(source) + (((((int) ceil(offset + i + phase) % (int) ceil(samplerate / frequency)) * 2.0 * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate) - 1.0) * volume)));
      }
      break;
    }

    source += star_synth_util->source_stride;
    i++;
  }
}

/**
 * ags_star_synth_util_compute_sawtooth:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR sawtooth synth.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_sawtooth(AgsStarSynthUtil *star_synth_util)
{
  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  switch(star_synth_util->format){
  case AGS_SOUNDCARD_SIGNED_8_BIT:
    {
      ags_star_synth_util_compute_sawtooth_s8(star_synth_util);
    }
    break;
  case AGS_SOUNDCARD_SIGNED_16_BIT:
    {
      ags_star_synth_util_compute_sawtooth_s16(star_synth_util);
    }
    break;
  case AGS_SOUNDCARD_SIGNED_24_BIT:
    {
      ags_star_synth_util_compute_sawtooth_s24(star_synth_util);
    }
    break;
  case AGS_SOUNDCARD_SIGNED_32_BIT:
    {
      ags_star_synth_util_compute_sawtooth_s32(star_synth_util);
    }
    break;
  case AGS_SOUNDCARD_SIGNED_64_BIT:
    {
      ags_star_synth_util_compute_sawtooth_s64(star_synth_util);
    }
    break;
  case AGS_SOUNDCARD_FLOAT:
    {
      ags_star_synth_util_compute_sawtooth_float(star_synth_util);
    }
    break;
  case AGS_SOUNDCARD_DOUBLE:
    {
      ags_star_synth_util_compute_sawtooth_double(star_synth_util);
    }
    break;
  case AGS_SOUNDCARD_COMPLEX:
    {
      ags_star_synth_util_compute_sawtooth_complex(star_synth_util);
    }
    break;
  }
}

/**
 * ags_star_synth_util_compute_triangle_s8:
 * @star_synth_util: the #AgsStarSynthUtil-struct
 * 
 * Compute STAR triangle synth of signed 8 bit data.
 *
 * Since: 7.4.0
 */
void
ags_star_synth_util_compute_triangle_s8(AgsStarSynthUtil *star_synth_util)
{
  gint8 *source, *tmp_source;

  guint samplerate;
  gdouble frequency;
  gdouble phase;
  gdouble lfo_frequency;
  gdouble lfo_depth;
  gdouble tuning;
  gdouble vibrato_gain;
  gdouble vibrato_lfo_depth;
  gdouble vibrato_lfo_freq;
  gdouble vibrato_tuning;
  guint offset;
  gdouble volume;
  guint i, i_stop;

  static const gdouble scale = 127.0;
  
  if(star_synth_util == NULL ||
     star_synth_util->source == NULL){
    return;
  }

  source = star_synth_util->source;

  samplerate = star_synth_util->samplerate;

  frequency = star_synth_util->frequency;
  phase = star_synth_util->phase;
  volume = star_synth_util->volume;

  lfo_frequency = star_synth_util->lfo_frequency;
  lfo_depth = star_synth_util->lfo_depth;
  tuning = star_synth_util->tuning;

  vibrato_gain = star_synth_util->vibrato_gain;
  vibrato_lfo_depth = star_synth_util->vibrato_lfo_depth;
  vibrato_lfo_freq = star_synth_util->vibrato_lfo_freq;
  vibrato_tuning = star_synth_util->vibrato_tuning;
  
  offset = star_synth_util->offset;
  
  i = 0;
  
#if defined(AGS_VECTORIZED_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    ags_v8double v_buffer, v_triangle;

    tmp_source = source;
    
    v_buffer = (ags_v8double) {
      (gdouble) *(tmp_source),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride),
      (gdouble) *(tmp_source += star_synth_util->source_stride)
    };
    
    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_triangle = (ags_v8double) {
	  ((((offset + i) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 1) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i + 1) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 2) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i + 2) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 3) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i + 3) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 4) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i + 4) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 5) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i + 5) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 6) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i + 6) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 7) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i + 7) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_triangle = (ags_v8double) {
	  ((((offset + i) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 1) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 2) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 3) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 4) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 5) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 6) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 7) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_triangle = (ags_v8double) {
	  ((((offset + i) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 1) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 2) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 3) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 4) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 5) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 6) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 7) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_triangle = (ags_v8double) {
	  ((((offset + i) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 1) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 2) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 3) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 4) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 5) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 6) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  ((((offset + i + 7) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0)
	};
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_triangle = (ags_v8double) {
	  (((offset + i + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  (((offset + i + 1 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  (((offset + i + 2 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  (((offset + i + 3 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  (((offset + i + 4 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  (((offset + i + 5 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  (((offset + i + 6 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0),
	  (((offset + i + 7 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0)
	};
      }
      break;
    default:
      v_triangle = (ags_v8double) {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0};
    }

    v_triangle *= scale;
    v_triangle *= volume;

    v_buffer += v_triangle;

    *(source) = (gint8) v_buffer[0];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[1];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[2];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[3];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[4];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[5];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[6];
    *(source += star_synth_util->source_stride) = (gint8) v_buffer[7];

    source += star_synth_util->source_stride;    
    
    i += 8;
  }
#elif defined(AGS_OSX_ACCELERATE_BUILTIN_FUNCTIONS)
  i_stop = star_synth_util->buffer_length - (star_synth_util->buffer_length % 8);
  
  for(; i < i_stop;){
    double ret_v_buffer[8], tmp0_ret_v_buffer[8], tmp1_ret_v_buffer[8];

    tmp_source = source;

    double v_buffer[] = {
      (double) *(tmp_source),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride),
      (double) *(tmp_source += star_synth_util->source_stride)};

    double v_triangle[8];

    switch(star_synth_util->lfo_oscillator_mode){
    case AGS_SYNTH_OSCILLATOR_SIN:
      {
	v_triangle[0] = ((((offset + i) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
	v_triangle[1] = ((((offset + i + 1) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i + 1) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 1) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
	v_triangle[2] = ((((offset + i + 2) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i + 2) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 2) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
	v_triangle[3] = ((((offset + i + 3) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i + 3) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 3) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
	v_triangle[4] = ((((offset + i + 4) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i + 4) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 4) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
	v_triangle[5] = ((((offset + i + 5) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i + 5) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 5) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
	v_triangle[6] = ((((offset + i + 6) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i + 6) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 6) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
	v_triangle[7] = ((((offset + i + 7) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) (((offset + i + 7) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + sin((offset + i + 7) * 2.0 * M_PI * lfo_frequency / samplerate) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SAWTOOTH:
      {
	v_triangle[0] = ((((offset + i) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
	v_triangle[1] = ((((offset + i + 1) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 1) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
	v_triangle[2] = ((((offset + i + 2) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 2) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
	v_triangle[3] = ((((offset + i + 3) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 3) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
	v_triangle[4] = ((((offset + i + 4) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 4) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
	v_triangle[5] = ((((offset + i + 5) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 5) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
	v_triangle[6] = ((((offset + i + 6) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 6) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
	v_triangle[7] = ((((offset + i + 7) + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7 + phase) * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((((int) ceil(offset + i + 7) % (int) ceil(samplerate / lfo_frequency)) * 2.0 * lfo_frequency / samplerate) - 1.0) * lfo_depth)) / samplerate)) / 2.0) * 2) - 1.0);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_TRIANGLE:
      { 
	v_triangle[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[1] = sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 1) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 1) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[2] = sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 2) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 2) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[3] = sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 3) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 3) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[4] = sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 4) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 4) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[5] = sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 5) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 5) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[6] = sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 6) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 6) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[7] = sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + (((offset + i + 7) * lfo_frequency / samplerate * 2.0) - ((int) ((double) ((int) ((offset + i + 7) * lfo_frequency / samplerate)) / 2.0) * 2) - 1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_SQUARE:
      {   
	v_triangle[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[1] = sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[2] = sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[3] = sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[4] = sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 4) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 4) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[5] = sin((gdouble) ((offset + i + 5) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 5) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 5) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[6] = sin((gdouble) ((offset + i + 6) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 6) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 6) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[7] = sin((gdouble) ((offset + i + 7) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 7) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 7) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= 0.0) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
      }
      break;
    case AGS_SYNTH_OSCILLATOR_IMPULSE:
      {
	v_triangle[0] = sin((gdouble) ((offset + i) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[1] = sin((gdouble) ((offset + i + 1) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 1) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 1) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[2] = sin((gdouble) ((offset + i + 2) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 2) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 2) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[3] = sin((gdouble) ((offset + i + 3) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0 * (vibrato_gain * sin((offset + i + 3) * 2.0 * M_PI * (vibrato_lfo_freq * (exp2(vibrato_tuning / 1200.0))) / samplerate) * vibrato_lfo_depth)) / 100.0) / 12.0) + ((sin((gdouble) (offset + i + 3) * 2.0 * M_PI * lfo_frequency / (gdouble) samplerate) >= sin(2.0 * M_PI * 3.0 / 5.0)) ? 1.0: -1.0) * lfo_depth)) / (gdouble) samplerate);
	v_triangle[4] = sin((gdouble) ((offset + i + 4) + phase) * 2.0 * M_PI * (frequency * exp2((((tuning + 100.0